module net.BurtonRadons.dig.common.listBox;

private import net.BurtonRadons.dig.platform.canvas;

/** A list of items that can be selected from, with a set of columns.
  */

class ListBox : Canvas
{
    private import net.BurtonRadons.dig.platform.base;
    private import net.BurtonRadons.dig.platform.control;
    
    /** An individual column. */
    class Column
    {
        ListBox control; /**< Link back to the list box. */
        char [] name; /**< Name of the column. */
        bit reversed; /**< If true, use reversed sorting order. */
        float width; /**< Current width in pixels. */
        
        /** Sort the rows on this column. */
        void sort ()
        {
            sort (reversed);
        }
        
        /** Sort the rows on this column.
          *
          * @param reversed If true, reverse the sorting order.
          */
          
        void sort (bit reversed)
        {
            this.reversed = reversed;
            control.digCommonSortColumn (this);
        }

        /** Return the index of this column when created. */
        int columnIndex ()
        {
            for (int c; c < control.digCommonColumns.length; c ++)
                if (this === control.digCommonColumns [c])
                    return c;
            assert (0);
        }

        /** Return the ordered index of this column. */
        int orderedIndex ()
        {
            for (int c; c < control.digCommonOrdered.length; c ++)
                if (this === control.digCommonOrdered [c])
                    return c;
            assert (0);
        }

        /** Return the x offset of this column. */
        int offsetx ()
        {
            int x = 0;

            for (int c; ; c ++)
            {
                assert (c < control.digCommonOrdered.length);
                if (this === control.digCommonOrdered [c])
                    return x;
                x += control.digCommonOrdered [c].width;
            }
        }

        /** Return whether this x point is within the column. */
        bit insidex (int x)
        {
            x -= offsetx ();
            return (x >= 0 && x < width);
        }
    }

    /** An item in the list.  The row has a number of dispatchers that are
      * relayed by the list when the row is either under the mouse or the
      * keyboard focus.  For mouse messages, the event.x and event.y
      * parameters refer to the start of the row in the list; so (0, 0) is the
      * top-left of the row, not of the list itself.
      */
    class Row
    {
        ListBox control; /**< Link back to the list box. */
        
        Dispatcher onLButtonDown; /**< The left mouse button has been pressed down. */
        Dispatcher onLButtonUp; /**< The left mouse button has been released. */
        Dispatcher onMButtonDown; /**< The middle mouse button has been pressed down. */
        Dispatcher onMButtonUp; /**< The middle mouse button has been released. */
        Dispatcher onRButtonDown; /**< The right mouse button has been pressed down. */
        Dispatcher onRButtonUp; /**< The right mouse button has been released. */
        Dispatcher onMouseMove; /**< The mouse has moved. */
        Dispatcher onMouseOver; /**< The mouse has moved over the row. */
        Dispatcher onMouseLeave; /**< The mouse is no longer over the row. */
        Dispatcher onFocus; /**< The row now has keyboard focus. */
        Dispatcher onLostFocus; /**< The row has lost keyboard focus. */
        Dispatcher onSelect; /**< The row has been selected. */
        Dispatcher onUnselect; /**< The row is no longer selected. */
        Dispatcher onChar; /**< The row is the keyboard focus and the user has pressed a key. */
        Dispatcher onKeyDown; /**< The row is the keyboard focus and the user has pressed a key. */
        Dispatcher onKeyUp; /**< The row is the keyboard focus and the user has pressed a key. */
        
        /** A change has been made to the row that will affect sorting.  By default this sorts the whole list. */
        void sortChange ()
        {
            control.sort ();
        }
        
        /** Return whether this is part of the selection. */
        bit selected ()
        {
            return this in control.digCommonSelection;
        }
        
        /** Return the row's zero-based index in the list box, or -1 if it's not in it. */
        int index ()
        {
            if (!control)
                return -1;
            for (int c; c < control.digCommonRows.length; c ++)
                if (control.digCommonRows [c] === this)
                    return c;
                
            return -1;
        }
        
        /** Return the vertical start of the row in the list box or -1 if it's not visible. */
        int offsety ()
        {
            int y = control.columnHeaderHeight ();
            int max = control.height ();
            
            for (int c = imax (control.vscrollPoint (), 0); c < control.digCommonRows.length && y < max; c ++)
            {
                if (control.digCommonRows [c] === this)
                    return y;
                y += control.digCommonRows [c].height ();
            }
            
            return -1;
        }

        /** Compare this row/column with another row/column.
          *
          * @return Positive if this row should be sorted later than the other row,
          *     Negative if this row should be sorted sooner than the other row,
          *     or 0 if they are not ordered.
          */
        abstract int compare (Row other, Column column);
        
        /** Draw a std.math.single column at the specified location.  The default draws
          * nothing.
          *
          * @param column The column to display.
          * @param x The horizontal coordinate to start the column.
          * @param y The vertical coordinate to start the column.
          * @param width The width in pixels of the space the column is given.
          */
          
        void displayColumn (Column column, int x, int y, int width)
        {
        }

        /** Display this row in the canvas at the specified location.  The default
          * calls displayFocus on the whole row, sets the text color to black or
          * white depending upon whether the row is selected, and then calls
          * displayColumn on each column.
          *
          * @param x The horizontal coordinate to start the row.
          * @param y The vertical coordinate to start the row.
          * @param selected True if this item has been selected.
          * @param focus True if this item is the keyboard focus.
          * @param enabled True if the control is enabled.
          */
        void display (int x, int y, bit selected, bit focus, bit enabled)
        {
            displayFocus (x, y, x + control.visualWidth (), y + height (), selected, focus, enabled);
            
            if (selected)
                control.textColor (Color.White);
            else
                control.textColor (Color.Black);
                
            for (int c; c < control.orderedColumnCount (); c ++)
            {
                Column column = control.orderedColumn (c);
                int ex = imin (control.visualWidth (), x + column.width);
                
                displayColumn (column, x, y, ex - x);
                x = ex;
            }
        }
        
        /** Display a selection, focus, and enabled box. */
        void displayFocus (int sx, int sy, int ex, int ey, bit selected, bit focus, bit enabled)
        {
            if (selected)
            {
                control.penClear ();
                if (enabled)
                    control.brushColor (49, 106, 197);
                else
                    control.brushColor (128, 128, 128);
                control.rect (sx, sy, ex, ey);
                control.textColor (Color.White);
            }
            
            if (focus)
                control.drawbox (sx, sy, ex, ey, control.DB.Focus, false);
        }

        /** Return the height of the row in pixels.
          */
          
        abstract int height ();

        override int opCmp (Object other)  /* operator overloading for  comparison */
        {
            for (int c; c < control.digCommonSorted.length; c ++)
            {
                Column column = control.digCommonSorted [c];
                int result = compare (cast (Row) other, column);

                if (column.reversed)
                    result = -result;
                if (result)
                    return result;
            }

            return 0;
        }
        
        /** Remove this row from the list. */
        void remove ()
        {
            if (control)
                control.remove (this);
        }
    }

    /** A row with text columns.  This simplified version of row is appropriate
      * if you don't have any non-text columns.  Columns that are too long for
      * their space are printed using ellipses.
      */
      
    class TextRow : Row
    {
        char [] [] list; /**< Column values. */

        /** Assign the list, non-copying. */
        this (char [] [] list)
        {
            this.list = list;
        }
        
        /** Return a column's text value. */
        char [] columnText (Column column)
        {
            int index = column.columnIndex ();
            
            if (index >= list.length)
                return null;
            return list [index];
        }

        override int compare (Row other, Column column)
        {
            if (cast (TextRow) other)
                return std.string.cmp (columnText (column), (cast (TextRow) other).columnText (column));
            return 0;
        }

        override int height ()
        {
            return control.textHeight ();
        }
        
        override void displayColumn (Column column, int x, int y, int width)
        {
            control.textPrintEllipses (x, y, width, 0, columnText (column));
        }
    }

    this (Control parent)
    {
        super (parent);
        onPaint.add (&digCommonDoPaint);
        onMouseMove.add (&digCommonDoMouseMove);
        onMouseLeave.add (&digCommonDoMouseLeave);
        onLButtonDown.add (&digCommonDoLButtonDown);
        onLButtonUp.add (&digCommonDoLButtonUp);
        onSizeChanged.add (&digCommonDoSizeChanged);
        onMButtonDown.add (&digCommonDoMButtonDown);
        onMButtonUp.add (&digCommonDoMButtonUp);
        onRButtonDown.add (&digCommonDoRButtonDown);
        onRButtonUp.add (&digCommonDoRButtonUp);
        onMouseWheel.add (&digCommonDoMouseWheel);
        onLostFocus.add (&digCommonDoLostFocus);
        onChar.add (&digCommonDoChar);
        onKeyDown.add (&digCommonDoKeyDown);
        onKeyUp.add (&digCommonDoKeyUp);
        pad (0, 0);
        
        bind ("Focus-Up", &funcFocusUp);
        bind ("Focus-Down", &funcFocusDown);
        bind ("Focus-Home", &funcFocusFirst);
        bind ("Focus-End", &funcFocusLast);
        
        bind ("Focus-Shift-Up", &funcFocusExtendUp);
        bind ("Focus-Shift-Down", &funcFocusExtendDown);
        bind ("Focus-Shift-Home", &funcFocusExtendFirst);
        bind ("Focus-Shift-End", &funcFocusExtendLast);
        
        bind ("Focus-Space", &funcFocusSelectExact);
        noSelection (true);
        vscroll (true);
    }
    
/** @name Bindable Functions
  * These methods are intended to be used for binding or for use in binding.
  */
    
/**@{*/
    
    /** Change the focus by a number of cells.  If there is no focus currently, it sets the focus to the first
      * or the last depending upon whether the delta's negative or positive, respectively.  Out-of-range
      * values are saturated.  Scrolls to continue to show the focus.
      */
      
    void funcFocusDelta (int delta)
    {
        if (!rowCount ())
            return;
        
        if (rowFocus ())
            setFocus (row (imid (0, rowFocus ().index () + delta, rowCount () - 1)));
        else if (delta < 0)
            setFocus (row (rowCount () - 1));
        else
            setFocus (row (0));
            
        if (!multipleSelection () || !stickySelection ())
        {
            selectBatch ();
            selectNone ();
            funcFocusSelect ();
            selectBatchFlush ();
        }
    }
    
    /** Change the focus by a number of cells, extending the selection.  If there is no focus currently, it
      * sets the focus to the first or last row depending upon whether the delta's negative or not.  Out-
      * of-range values are saturated.  Then it selects all the rows between the current and the last
      * position.  Scrolls to continue to show the focus.
      */
      
    void funcFocusExtendDelta (int delta)
    {
        if (!multipleSelection ())
        {
            funcFocusDelta (delta);
            return;
        }
        
        if (!rowCount ())
            return;
        
        int last;
        
        if (rowFocus ())
        {
            last = rowFocus ().index ();
            setFocusWithoutChangingReference (row (imid (0, rowFocus ().index () + delta, rowCount () - 1)));
        }
        else if (delta < 0)
            setFocusWithoutChangingReference (row (last = rowCount () - 1));
        else
            setFocusWithoutChangingReference (row (last = 0));
            
        int min = digCommonReference.index (), max = rowFocus ().index ();
        
        if (min > max)
            min = rowFocus ().index (), max = digCommonReference.index ();
     
        selectBatch ();
        if (!stickySelection ())
            selectNone ();
        for (int c = min; c <= max; c ++)
            selectAdd (row (c));
        selectBatchFlush ();
    }
    
    /** Move the focus up a line or to the end if there is no focus.  Scroll to continue to show the focus. */
    void funcFocusUp ()
    {
        funcFocusDelta (-1);
    }
    
    /** Extend the focus up a line. */
    void funcFocusExtendUp ()
    {
        funcFocusExtendDelta (-1);
    }
    
    /** Move the focus down a line or to the beginning if there is no focus.  Scroll to continue to show the focus. */
    void funcFocusDown ()
    {
        funcFocusDelta (+1);
    }
    
    /** Extend the focus down a line. */
    void funcFocusExtendDown ()
    {
        funcFocusExtendDelta (+1);
    }
    
    /** Move the focus to the beginning of the list. */
    void funcFocusFirst ()
    {
        funcFocusDelta (-rowCount ());
    }
    
    /** Extend the selection to the beginning of the list. */
    void funcFocusExtendFirst ()
    {
        funcFocusExtendDelta (-rowCount ());
    }
    
    /** Move the focus to the end of the list. */
    void funcFocusLast ()
    {
        funcFocusDelta (+rowCount ());
    }
    
    /** Extend the selection to the end of the list. */
    void funcFocusExtendLast ()
    {
        funcFocusDelta (+rowCount ());
    }
    
    /** Select the row under the keyboard focus.  If there is no focus or if no selections are
      * allowed, this is ignored.  If this is a std.math.single selection list box, the focus is changed.
      * If this is a multiple selection list box, this selection value is toggled.
      */
    void funcFocusSelect ()
    {
        if (!rowFocus () || noSelection ())
            return;
        selectAction (rowFocus ());
    }
    
    /** funcFocusSelect can be called under several circumstances.  funcFocusSelectExact
      * is only called under direct user control.
      */
      
    void funcFocusSelectExact ()
    {
        funcFocusSelect ();
    }
    
/** @} */
    
    /** Return whether this doesn't allow selection changes. */
    bit noSelection ()
    {
        return digCommonNoSelection;
    }
    
    /** Assign whether this doesn't allow selection changes. */
    void noSelection (bit value)
    {
        digCommonNoSelection = value;
    }
    
    /** Return whether this allows multiple selections. */
    bit multipleSelection ()
    {
        return digCommonMultipleSelection;
    }
    
    /** Assign whether this allows multiple selections.  This doesn't affect the selection,
      * so turning it off with multiple rows currently selected has no effect.
      */
      
    void multipleSelection (bit value)
    {
        digCommonMultipleSelection = value;
    }
    
    /** Return whether to use sticky selection.  When set with a multiple selection list box,
      * clicking on a row with the mouse will toggle whether this item is selected.  When
      * reset, clicking on a row will clear the selection to only be that row.  For std.math.single
      * selection list boxes or disabled list boxes, this setting is not relevant.
      */
      
    bit stickySelection ()
    {
        return digCommonStickySelection;
    }
    
    /** Assign whether to use sticky selection. */
    
    void stickySelection (bit value)
    {
        digCommonStickySelection = value;
    }
    
    /** Return whether the column header should be displayed.  By default this
      * is true.  Showing the column header allows changing column sizes,
      * ordering, and every other user interface operation for list boxes, so
      * hiding the header is usually only advisable if there's only one column.
      */
      
    bit columnHeaderShown ()
    {
        return digCommonColumnHeaderShown;
    }
    
    /** Assign whether the column header should be displayed.  If this value
      * is different from its previous setting, the control is painted.
      */
      
    void columnHeaderShown (bit value)
    {
        if (value == columnHeaderShown ())
            return;
        digCommonColumnHeaderShown = value;
        paint ();
    }
    
    /** Set the keyboard focus, scroll to show it, and paint as needed. */
    void setFocus (Row row)
    {
        if (digCommonFocus !== row && !vscrollToShowRow (row))
        {
            paintRow (row);
            paintRow (digCommonFocus);
        }
        
        Row old = digCommonFocus;

        digCommonFocus = row;
        digCommonReference = row;

        if (old !== row)
        {
            if (row)
                row.onFocus.notify ();
            if (old)
                old.onLostFocus.notify ();
        }
    }
    
    /** Set the keyboard focus without affecting the reference, which is used for shift-click. */
    void setFocusWithoutChangingReference (Row row)
    {
        Row oldReference = digCommonReference;
        
        setFocus (row);
        digCommonReference = oldReference;
    }
    
/** @name Row Management
  *
  * These methods control rows.
  */
  
/** @{ */
    
    /** Vertical scroll if needed to show this row, and return whether it did need to scroll. */
    bit vscrollToShowRow (Row row)
    {
        if (row === null)
            return false;
        
        int index = row.index ();
        int height = this.height () - columnHeaderHeight ();
        
        /* Scroll up if needed. */
        if (index < vscrollPoint ())
        {
            vscrollPoint (index);
            paint ();
            return true;
        }
        
        /* Bail out if fully in view. */
        int y = row.offsety ();
        
        if (y >= 0 && y + row.height () <= this.height ())
            return false;
        
        /* Back up in rows until we can't go back anymore. */
        int h = height, c;
        
        for (c = index; c >= 0; c --)
        {
            Row item = this.row (c);
            
            h -= item.height ();
            if (h <= 0)
            {
                c ++;
                break;
            }
        }
        
        if (c != vscrollPoint ())
        {
            vscrollPoint (c);
            paint ();
            return true;
        }
        
        return false;
    }
    
    /** Sort the list and repaint the control. */
    void sort ()
    {
        digCommonRows.sort;
        if (!vscrollToShowRow (rowFocus ()))
            paint ();
    }

    /** Add a row, does not paint. */
    void add (Row row)
    {
        digCommonRows ~= row;
        row.control = this;
    }
    
    /** Add a text row with a std.math.single column, does not paint. */
    void addText (char [] a)
    {
        char [] [] list = new char [] [1];
        
        list [0] = a;
        add(new TextRow (list));
    }

    /** Add a text row with two columns, does not paint. */
    void addText (char [] a, char [] b)
    {
        char [] [] list = new char [] [2];

        list [0] = a;
        list [1] = b;
        add(new TextRow (list));
    }
    
    /** Insert the row after the indexed row, and repaint the control. */
    void insertAfter (Row row, int index)
    {
        insertBefore (row, index + 1);
    }
    
    /** Insert the row before the indexed row, and repaint the control. */
    void insertBefore (Row row, int index)
    {
        digCommonRows ~= null;
        for (int c = digCommonRows.length - 1; c > index; c --)
            digCommonRows [c] = digCommonRows [c - 1];
        digCommonRows [index] = row;
        row.control = this;
        paint ();
    }
    
    /** Index the array of rows.  Out-of-range values will be bounds checked. */
    Row row (int index)
    {
        return digCommonRows [index];
    }
    
    /** Return the number of rows in the list. */
    int rowCount ()
    {
        return digCommonRows.length;
    }
    
    /** Return the current row with keyboard focus or null if there is none. */
    Row rowFocus ()
    {
        return digCommonFocus;
    }
    
    /** Paint a std.math.single row. */
    void paintRow (Row row)
    {
        if (row === null)
            return;
        int y = row.offsety ();
        
        if (y >= 0)
            paintRegion (0, y, width (), y + row.height () + 1);
    }
    
    /** Paint from the row to the end of the control. */
    void paintFromRow (Row row)
    {
        if (row === null)
            return;
        int y = row.offsety ();
        
        if (y >= 0)
            paintRegion (0, y, width (), height ());
    }
    
    /** Paint from the indexed row to the end of the control. */
    void paintFromRow (int index)
    {
        if (index < 0 || index >= rowCount ())
            return;
        paintFromRow (row (index));
    }
    
    /** Determine the row this Y coordinate is over or return null if it is out-of-range. */
    Row findRow (int y)
    {
        y -= columnHeaderHeight ();
        if (y < 0)
            return null;
        for (int c = vscrollPoint (); c < digCommonRows.length; c ++)
        {
            y -= digCommonRows [c].height ();
            if (y < 0)
                return digCommonRows [c];
        }
        
        return null;
    }
    
    /** Delete all the rows in the list, then paint the control. */
    void empty ()
    {
        if (rowCount ())
            paint ();
        digCommonRows = null;
    }
    
    /** Remove a std.math.single row from the list, then paint the control. */
    void remove (Row row)
    {
        removeFilter (delegate bit (Row compare) { return row !== compare; });
    }
    
    /** Remove a number of rows from the list by iterating over the rows and
      * calling func on each of them.  If it returns true, the row remains in the
      * list, but if it returns false, the row is removed.  This paints the control
      * if any rows were removed.
      *
      * @param func The function to call on each row.  This will be called with a
      * row from the list and should return true to keep the row in the list or false
      * to remove it.
      */
    void removeFilter (bit delegate (Row row) func)
    {
        Row [] rows = digCommonRows;
        int c, d;
        
        for (int c, d; ; c ++)
            if (c >= rows.length)
            {
                digCommonRows = rows [0 .. d];
                return;
            }
            else if (func (rows [c]))
                rows [d ++] = rows [c];
    }
    
/** @} */
    
/** @name Column Management
  *
  * These methods create, access, and delete columns.
  */
  
/** @{ */

    /** Append a column to the list of columns. */
    Column addColumn (char [] name, int width)
    {
        Column column = new Column;

        column.control = this;
        column.name = name;
        column.reversed = false;
        column.width = width;
        digCommonColumns ~= column;
        digCommonOrdered ~= column;
        digCommonColumnCount ++;

        int w = 0;

        for (int c; c < digCommonColumnCount; c ++)
            w += digCommonOrdered [c].width;
        this.suggestWidth (w);

        return column;
    }

    /** Return the column this index is over. */
    Column findColumn (int x)
    {
        for (int c; c < digCommonColumnCount; c ++)
        {
            Column column = digCommonOrdered [c];

            x -= column.width;
            if (x < 0)
                return column;
        }

        return null;
    }

    /** Index the columns array. */
    Column orderedColumn (int index)
    {
        return digCommonOrdered [index];
    }

    /** Return the number of visible ordered columns. */
    int orderedColumnCount ()
    {
        return digCommonColumnCount;
    }
    
    /** Return the visible height of the column header in pixels. */
    int columnHeaderHeight ()
    {
        if (!columnHeaderShown ())
            return 0;
        return textHeight () + 4;
    }
    
    /** Paint a std.math.single column's header. */
    void paintColumnHeader (Column column)
    {
        if (!column || !columnHeaderShown ())
            return;
        
        int offset = column.offsetx ();
        
        paintRegion (offset, 0, offset + column.width, columnHeaderHeight ());
    }
    
/** @} */
    
/** @name Selection Management
  *
  * These methods manage the current selection.
  */
  
/** @{ */
    
    /** Change the current selection.  If no selection is allowed, this is ignored.  If this is a
      * std.math.single-selection list box, it assigns the selection.  If this is a multiple-selection list
      * box, it toggles whether this item is selected.  This notifies onSelectionAdd and
      * onSelectionRemove of the change.
      */
      
    void selectAction (Row row)
    {
        if (row === null || noSelection ())
            return;
        if (!multipleSelection ())
        {
            if (digCommonSelection.length)
                digCommonNotifySelectionRemove (digCommonSelection.keys [0]);
            digCommonNotifySelectionAdd (row);
        }
        else if (digCommonMultipleSelection && row in digCommonSelection)
            digCommonNotifySelectionRemove (row);
        else
            digCommonNotifySelectionAdd (row);
    }
    
    /** Add to the selection.  If no selection is allowed, this is ignored.  If this is a
      * std.math.single-selection list box, it assigns the selection.  If this is a multiple-
      * selection list box, this item is selected.  This notifies onSelectionAdd of any
      * change.
      */
      
    void selectAdd (Row row)
    {
        if (row === null || noSelection () || row.selected ())
            return;
        if (!multipleSelection () && digCommonSelection.length)
            digCommonNotifySelectionRemove (digCommonSelection.keys [0]);
        digCommonNotifySelectionAdd (row);
    }
    
    /** Clear the selection.  This notifies onSelectionRemove of the changes. */
    void selectNone ()
    {
        Row [] keys = digCommonSelection.keys.dup;
        
        for (int c; c < keys.length; c ++)
        {
            paintRow (keys [c]);
            digCommonNotifySelectionRemove (keys [c]);
        }
    }
    
    /** Start batching a selection change to minimise the number of items
      * actually notified or repainted.  Usage is like this:
      *
      * @code
      * control.selectBatch (); // Start batching.
      * control.selectAdd (item1); // Notifications and repainting will be avoided.
      * control.selectBatchFlush (); // Notify of changes and repaint as needed.
      * @endcode
      *
      */
      
    void selectBatch ()
    {
        if (digCommonSelectBatching)
            selectBatchFlush ();
        digCommonSelectionBatch = null;
        Row [] keys = digCommonSelection.keys;
        bit [] values = digCommonSelection.values;
        for (int c; c < keys.length; c ++)
            digCommonSelectionBatch [keys [c]] = values [c];
        digCommonSelectBatching = true;
    }
    
    /** Flush all batched changes to the selection and stop batching. */
    
    void selectBatchFlush ()
    {
        if (!digCommonSelectBatching)
            return;
        Row [] keys;
        bit [Row] batch = digCommonSelectionBatch;
        
        digCommonSelectBatching = false;

        keys = digCommonSelection.keys;        
        for (int c; c < keys.length; c ++)
            if (!(keys [c] in batch))
                digCommonNotifySelectionAdd (keys [c]);
                
        keys = batch.keys;
        for (int c; c < keys.length; c ++)
            if (!(keys [c] in digCommonSelection))
                digCommonNotifySelectionRemove (keys [c]);
    }
    
/** @} */
    
    /** The methods in this list are notified when an item has been added
       * to the selection.  The listbox is a pointer back to the list box, while row
       * is the row added.  These methods are called in index order.
       */
       
    void delegate (ListBox listbox, Row row) [] onSelectionAdd;
       
    /** The methods in this list are notified when an item has been removed
      * from the selection.  The listbox is a pointer back to the list box, while
      * row is the row added.  These methods are called in index order.
      */
      
    void delegate (ListBox listbox, Row row) [] onSelectionRemove;

/+
#ifdef DoxygenMustSkipThis
+/

    void digCommonNotifySelectionAdd (Row row)
    {
        if (row === null)
            return;
        digCommonSelection [row] = true;
        if (digCommonSelectBatching)
            return;
        paintRow (row);
        for (int c; c < onSelectionAdd.length; c ++)
            onSelectionAdd [c] (this, row);
        row.onSelect.notify ();
    }
    
    void digCommonNotifySelectionRemove (Row row)
    {
        if (row === null)
            return;
        delete digCommonSelection [row];
        if (digCommonSelectBatching)
            return;
        paintRow (row);
        for (int c; c < onSelectionRemove.length; c ++)
            onSelectionRemove [c] (this, row);
        row.onUnselect.notify ();
    }
    
    void digCommonSetupVScroll ()
    {
        int y = columnHeaderHeight ();
        int c;
        
        for (c = 0; ; c ++)
        {
            int i = vscrollPoint () + c;
            
            if (i >= rowCount ())
                break;
            y += row (i).height ();
            if (y > height ())
                break;
        }
        
        vscrollRangeAndPage (0, digCommonRows.length - 1, c);
    }

    void digCommonDoPaint ()
    {
        beginPaint ();
        clear (backgroundColor ());

        digCommonSetupVScroll ();
        
        bit captor = isCaptor ();
        int y;
        
        /* Draw the column header. */
        if (columnHeaderShown ())
        {
            for (int c, x; c < digCommonColumnCount; c ++)
            {
                Column column = digCommonOrdered [c];
                bit over = (digCommonMouseOverColumn === column) && (digCommonTask != "resize");
                int ex = imin (visualWidth (), x + column.width);
    
                listboxColumnHeaderDraw (x, 0, ex, textHeight () + 4, over, over && captor, column.name);
                if (digCommonSorted.length && column == digCommonSorted [0])
                    listboxColumnArrowDraw (ex - 14, 0, ex - 6, textHeight () + 4, column.reversed);
    
                x += column.width;
            }
            
            y += columnHeaderHeight ();
        }

        /* Draw the background fill. */
        for (int c, x; c < digCommonColumnCount; c ++)
        {
            Column column = digCommonOrdered [c];
            int ex = imin (visualWidth (), x + column.width);

            listboxItemDraw (x, y, ex, this.height (), digCommonSorted.length && column === digCommonSorted [0]);
            x += column.width;
        }

        /* Now draw the rows. */
        for (int c; ; c ++)
        {
            int i = vscrollPoint () + c;

            if (i >= digCommonRows.length)
                break;

            Row row = digCommonRows [i];
            int height = row.height ();
            
            row.display (3, y, row in digCommonSelection, digCommonFocus === row, true);
            y += height;
            if (y > this.height ())
                break;
        }

        endPaint ();
    }

    bit digCommonIsOverEdge (int mx)
    {
        Column column = digCommonMouseOverColumn;

        if (column === null)
            return false;
        if ((mx - column.offsetx () < 2 && column != digCommonOrdered [0])
         || ((column.offsetx () + column.width) - mx < 4 && column != digCommonOrdered [digCommonColumnCount - 1]))
            return true;
        return false;
    }

    void digCommonDoMouseMove (Event event)
    {
        Column previous = digCommonMouseOverColumn;
        Row previousRow = digCommonMouseOverRow;

        if (isCaptor ())
        {
            if (digCommonTask == "resize")
            {
                float old = previous.width;
                float delta;

                previous.width = imid (0, previous.width - event.deltax, width () - previous.offsetx ());
                delta = old - previous.width;

                int index = previous.orderedIndex ();
                int count = orderedColumnCount ();

                for (int c = index + 1; c < count; c ++)
                    digCommonOrdered [c].width += delta / (count - index - 1);
                paint ();
            }

            return;
        }

        cursor (Cursor.Arrow);

        if (event.y < columnHeaderHeight ())
        {
            digCommonMouseOverColumn = findColumn (event.x);
            if (digCommonIsOverEdge (event.x))
                cursor (Cursor.SizeEW);
            digCommonMouseOverRow = null;
        }
        else
        {
            digCommonMouseOverColumn = null;
            digCommonMouseOverRow = findRow (event.y);
        }
        
        digCommonMouseOverRowCheck (previousRow);
        
        if (previous !== digCommonMouseOverColumn)
        {
            paintColumnHeader (previous);
            paintColumnHeader (digCommonMouseOverColumn);
        }
    }
    
    void digCommonMouseOverRowCheck (Row previous)
    {
        if (previous === digCommonMouseOverRow)
            return;
        if (previous)
            previous.onMouseLeave.notify ();
        if (digCommonMouseOverRow)
            digCommonMouseOverRow.onMouseOver.notify ();
    }

    void digCommonDoMouseLeave ()
    {
        if (isCaptor ())
            return;
        if (digCommonMouseOverColumn !== null)
        {
            paintColumnHeader (digCommonMouseOverColumn);
            digCommonMouseOverColumn = null;
        }

        Row previousRow = digCommonMouseOverRow;        
        digCommonMouseOverRow = null;
        digCommonMouseOverRowCheck (previousRow);
    }

    void digCommonDoLButtonDown (Event event)
    {
        captureMouse ();
        
        if (event.y > columnHeaderHeight ())
        {
            Row row = findRow (event.y);

            if (!row)
                return;
            Event subEvent = event;
            
            subEvent.x -= 3;
            subEvent.y -= row.offsety ();
            row.onLButtonDown.notify (subEvent);
            
            if (event.shift && multipleSelection ())
            {
                selectBatch ();
                if (!event.control)
                    selectNone ();
                int min = row.index ();
                int max = digCommonReference.index ();
                
                if (min > max)
                    min = digCommonReference.index (), max = row.index ();
                    
                for (int c = min; c <= max; c ++)
                    selectAdd (this.row (c));
                makeFocus ();
                setFocusWithoutChangingReference (row);
                selectBatchFlush ();
            }
            else if (stickySelection () || !multipleSelection () || event.control)
            {
                makeFocus ();
                setFocus (row);
                selectAction (row);
            }
            else
            {
                makeFocus ();
                setFocus (row);
                selectBatch ();
                selectNone ();
                selectAdd (row);
                selectBatchFlush ();
            }
        }
        else if (digCommonIsOverEdge (event.x))
        {
            digCommonTask = "resize";
            if (event.x < digCommonMouseOverColumn.offsetx () + digCommonMouseOverColumn.width / 2)
                digCommonMouseOverColumn = digCommonOrdered [digCommonMouseOverColumn.orderedIndex () - 1];
        }
        else
        {
            digCommonTask = "sort";
            paintColumnHeader (digCommonMouseOverColumn);
        }
    }
    
    void digCommonSortColumn (Column column)
    {
        int offset = 0;
        
        for (int c = digCommonSorted.length - 1; c > 0; c --)
        {
            if (digCommonSorted [c] === column)
                offset = 1;
            digCommonSorted [c] = digCommonSorted [c - offset];
        }

        if (!offset && !(digCommonSorted.length && digCommonSorted [0] === column))
        {
            digCommonSorted.length = digCommonSorted.length + 1;
            for (int c = digCommonSorted.length - 1; c > 0; c --)
                digCommonSorted [c] = digCommonSorted [c - 1];
        }

        digCommonSorted [0] = column;

        sort ();
        paint ();
    }

    void digCommonDoLButtonUp (Event event)
    {
        releaseMouse ();

        if (digCommonTask == "sort" && digCommonMouseOverColumn !== null && digCommonMouseOverColumn.insidex (event.x))
        {
            if (digCommonSorted.length && digCommonSorted [0] == digCommonMouseOverColumn)
                digCommonMouseOverColumn.reversed = !digCommonMouseOverColumn.reversed;
            digCommonSortColumn (digCommonMouseOverColumn);
        }
        
        Row row = findRow (event.y);
        if (row)
        {
            Event subEvent = event;
            
            subEvent.x -= 3;
            subEvent.y -= row.offsety ();
            row.onLButtonUp.notify (subEvent);
        }

        digCommonTask = null;
    }

    void digCommonDoSizeChanged (Event e)
    {
        for (int c; c < digCommonColumnCount; c ++)
        {
            Column column = digCommonOrdered [c];

            column.width *= width () / (float) e.x;
        }
    }
    
    void digCommonDoMButtonDown (Event event)
    {
        Row row = findRow (event.y);
        
        if (row)
            row.onMButtonDown.notify (digCommonProcessEvent (row, event));
    }
    
    void digCommonDoMButtonUp (Event event)
    {
        Row row = findRow (event.y);
        
        if (row)
            row.onMButtonUp.notify (digCommonProcessEvent (row, event));
    }
    
    void digCommonDoRButtonDown (Event event)
    {
        Row row = findRow (event.y);
        
        if (row)
            row.onRButtonDown.notify (digCommonProcessEvent (row, event));
    }
    
    void digCommonDoRButtonUp (Event event)
    {
        Row row = findRow (event.y);
        
        if (row)
            row.onRButtonUp.notify (digCommonProcessEvent (row, event));
    }
    
    void digCommonDoLostFocus (Event event)
    {
        if (!digCommonFocus)
            return;
        digCommonFocus.onLostFocus.notify ();
        paintRow (digCommonFocus);
        digCommonFocus = null;
    }
    
    void digCommonDoChar (Event event)
    {
        if (rowFocus ())
            rowFocus ().onChar.notify (event);
    }
    
    void digCommonDoKeyDown (Event event)
    {
        if (rowFocus ())
            rowFocus ().onKeyDown.notify (event);
    }
    
    void digCommonDoKeyUp (Event event)
    {
        if (rowFocus ())
            rowFocus ().onKeyUp.notify (event);
    }
    
    /** Process the event for sending to the row. */
    Event digCommonProcessEvent (Row row, Event event)
    {
        event.x -= 3;
        event.y -= row.offsety ();
        return event;
    }
    
    void digCommonDoMouseWheel (Event event)
    {
        int offset = event.wheel * imax (1, vscrollPage () / 6);

        vscrollPoint (vscrollPoint () - offset);
        paint ();
    }
    
    char [] digCommonTask; /* The current task; null, "sort", "resize", "select-on", "select-off". */
    Column [] digCommonColumns; /* The original set of columns. */
    Column [] digCommonOrdered; /* The ordered set of columns. */
    Column [] digCommonSorted; /* The sorting order of columns. */
    Row [] digCommonRows; /* List of rows in the listbox. */
    int digCommonColumnCount; /* The number of columns to actually display. */
    Column digCommonMouseOverColumn; /**< The column that the mouse is currently over. */
    bit [Row] digCommonSelection; /**< Current selection, each value is true. */
    bit digCommonMultipleSelection; /**< Allows multiple selection. */
    bit digCommonNoSelection; /**< Don't allow any selections. */
    bit digCommonStickySelection; /**< With a multiple selection list box, clicking the selection without holding down any modifiers will not cause the selection to clear. */
    Row digCommonFocus; /**< Current keyboard focus or null. */
    Row digCommonReference; /**< Reference row for shift-clicking. */
    Row digCommonMouseOverRow; /**< The row the mouse is over. */
    bit digCommonSelectBatching; /**< Currently batching selection changes. */
    bit [Row] digCommonSelectionBatch; /**< Prior selection state before selectBatch. */
    bit digCommonColumnHeaderShown = true; /**< Whether the column header should be shown. */
    
/+
#endif
+/
}
